#include "drawarea.h"
#include <cmath>
#include <QFont>
#include <QDateTime>
#include <QPainterPath>
#include <QTimer>
#include <QTimeZone>

constexpr const char* const TIME_ZONE_NAMES[5] = {
    "local",
    "UTC",
    "Europe/Moscow",
    "Asia/Shanghai",
    "America/New_York"
};

DrawArea::DrawArea(QWidget *parent):
    QWidget{parent},
    xmin(-1.),
    xmax(1),
    ymin(-1.),
    ymax(1),
    radiusX(((xmax - xmin)/2.)*0.9),
    radiusY(((ymax - ymin)/2.)*0.9),
    longStroke(0.1),
    shortStroke(longStroke/3.),
    textShift(2.*longStroke),
    handHour(0.5),
    handMin(0.7),
    handSec(0.9),
    handHourColor(Qt::blue),
    handMinColor(Qt::darkGreen),
    handSecColor(Qt::red),
    initialUpdate(true),
    timeZoneID(0)
{
    timeZones[0] = QTimeZone();
    for (int i = 1; i < NUM_TIME_ZONES; ++i) {
        timeZones[i] = QTimeZone(TIME_ZONE_NAMES[i]);
    }
}

void DrawArea::paintEvent(QPaintEvent *) {
    QPainter painter(this);
    painter.fillRect(
        0, 0,
        width(), height(),
        QBrush(Qt::lightGray)
    );
    painter.setRenderHint(QPainter::Antialiasing);
    drawFace(&painter);
    drawHands(&painter);
    if (initialUpdate) {
        initialUpdate = false;
        animate();
    }
}

void DrawArea::drawFace(QPainter* painter) {
    int w = width();
    int h = height();
    if (w == 0 || h == 0)
        return;
    R2Point center(
        (xmin + xmax)/2.,
        (ymin + ymax)/2.
    );
    QPen blackPen(Qt::black, 3);
    QBrush whiteBrush(Qt::white);
    painter->setPen(blackPen);
    painter->setBrush(whiteBrush);
    QPointF pixelCenter = mapToPixels(center);
    QPointF pixRad = mapToPixels(
        R2Point(xmin + radiusX, ymax - radiusY)
    );
    double pixelRadiusX = pixRad.x();
    double pixelRadiusY = pixRad.y();
    painter->drawEllipse(
        pixelCenter, pixelRadiusX, pixelRadiusY
    );

    QFont digitsFont("Helvetica", 28, QFont::Bold);
    painter->setFont(digitsFont);
    QFontMetrics fontMetrics(digitsFont);

    double minuteAngle = M_PI*2./60.;
    double angle = M_PI/2. - minuteAngle;
    for (int i = 1; i <= 60; ++i) {
        R2Vector direction(
            cos(angle)*radiusX,
            sin(angle)*radiusY
        );
        if (i%5 == 0) {
            // Long stroke
            painter->drawLine(
                mapToPixels(center + direction*(1.-longStroke)),
                mapToPixels(center + direction)
            );

            QString hours= QString::number(i/5);
            QRect textRect = fontMetrics.boundingRect(hours);
            QPointF textPoint = QPointF(
                -(textRect.left() + textRect.right())/2.,
                // -textRect.top()
                -(textRect.top() + textRect.bottom())/2.
            );
            R2Vector textCorrection =
                    mapFromPixels(textPoint) -
                    R2Point(xmin, ymax);

            painter->drawText(
                mapToPixels(
                    center + direction*(1. - textShift)
                    + textCorrection
                ),
                hours
            );
        } else {
            // Short stroke
            painter->drawLine(
                mapToPixels(center + direction*(1.-shortStroke)),
                mapToPixels(center + direction)
            );
        }
        angle -= minuteAngle;
    }
}

void DrawArea::drawHands(QPainter* painter) {
    QDateTime local_time = QDateTime::currentDateTime();
    QDateTime dateTime = local_time;

    // QList<QByteArray> tzs =
    //     QTimeZone::availableTimeZoneIds();

    if (timeZoneID > 0) { // not the local time
        dateTime = local_time.toTimeZone(
            timeZones[timeZoneID]
        );
    }

    int hour = dateTime.time().hour();
    int minute = dateTime.time().minute();
    int second = dateTime.time().second();

    // Test
    // hour = 6; minute = 20; second = 40;

    double angleHour = M_PI/2. - (
        hour*(2.*M_PI)/12. +
        minute*(2.*M_PI)/(12.*60.) +
        second*(2.*M_PI)/(12.*60.*60.)
    );

    double angleMin = M_PI/2. - (
        minute*(2.*M_PI)/60. +
        second*(2.*M_PI)/(60.*60)
    );

    double angleSec = M_PI/2. - (
        second*(2.*M_PI)/60.
    );

    drawHand(painter, angleHour, handHour, handHourColor);
    drawHand(painter, angleMin, handMin, handMinColor);
    drawHand(painter, angleSec, handSec, handSecColor);
}

void DrawArea::drawHand(
    QPainter* painter,
    double angle,
    double handLength,
    QColor handColor
) {
    R2Vector direction(
        cos(angle)*radiusX,
        sin(angle)*radiusY
    );
    direction *= handLength;
    // Longer hand is narrower
    double w = 0.1*(1./(5.*handLength*handLength));
    R2Point center(
        (xmin + xmax)/2., (ymin + ymax)/2.
    );
    R2Vector oppDirection = direction*(-0.25);
    R2Vector ortDirection = direction.normal()*(-w);
    R2Point p0 = center + oppDirection;
    R2Point p1 = center - ortDirection;
    R2Point p2 = center + direction;
    R2Point p3 = center + ortDirection;

    QPainterPath path;
    path.moveTo(mapToPixels(p0));
    path.lineTo(mapToPixels(p1));
    path.lineTo(mapToPixels(p2));
    path.lineTo(mapToPixels(p3));
    path.closeSubpath();

    QPen pen(handColor);
    painter->setPen(pen);
    QBrush brush(handColor);
    painter->setBrush(brush);
    painter->drawPath(path);
}

void DrawArea::animate() {
    QTimer::singleShot(
        1000, this,
        &DrawArea::redraw
    );
}

void DrawArea::redraw() {
    update();
    animate();
}

void DrawArea::setTimeZone(int id) {
    if (0 <= id  && id < NUM_TIME_ZONES) {
        timeZoneID = id;
        update();
    }
}
